#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <time.h>
#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#endif
#include <ctype.h>

#ifdef WIN32
#include <windows.h>
#endif

#include "freetype/freetype.h"
#include "freetype/ftxpost.h"     /* we are in the FreeType package tree */

#include "gbk2uni.h"          /* add support GBK Chinese Unicode ttf fonts */
#include "big52uni.h"

char rcsid[] = "$Id: ttf2pfb.c,v 1.2 2002/07/27 12:58:58 wy Exp $";
extern unsigned short pid, eid;

#ifndef TRUE
#define TRUE  1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef FAILURE
#define FAILURE -1
#endif


#define LINELEN 40  /* max # of columns of code range file */
#define NAMELEN 80  /* max length of name used from LookUp */


/*
 * globals
 */

static int   code_idx[256];

#ifndef UShort
typedef unsigned short UShort;
#endif
#ifndef UChar
typedef unsigned char  UChar;
#endif

#define TT_Flag_On_Curve 1

/* default platform and encoding ID's. */
static char* prog = "ttf2pfb";             /* program name */


/* for orthogonality with fprintf */
#define Fputs(_string_) fprintf(out, "%s\n", _string_)


/* Postscript font related defines and functions */

#define Coord(x) (int)(x)
#define PS_LastPt(x, y) lastpsx = (x); lastpsy = (y)
#define PS_Moveto(x, y) fprintf(out, "%d %d rmoveto\n", Coord((x) - lastpsx), Coord((y) - lastpsy)); PS_LastPt((x), (y))
#define PS_Lineto(x, y) fprintf(out, "%d %d rlineto\n", Coord((x) - lastpsx), Coord((y) - lastpsy)); PS_LastPt((x), (y))


/*
 * Freetype globals.
 */

TT_F26Dot6 lastpsx, lastpsy;
static TT_Engine   engine;
static TT_Face     face;
static TT_Instance instance;
static TT_Glyph    glyph;
static TT_CharMap  cmap;
static TT_Error    error;
static TT_Post     post;
static TT_Outline         outline;
static TT_Glyph_Metrics   metrics;
static TT_Face_Properties properties;


/*
 * Data structures defined for encoding vectors
 */

typedef struct _EVHigh
{
	UShort start, end;
} EVHigh;

typedef struct _EVLow
{
	UChar start, end;
} EVLow;

typedef struct _EVcRange
{
	EVHigh high;
	UShort numLowRanges;
	EVLow* low;
} EVcRange;

typedef struct _EncVec
{
	UShort    numCodeRanges;
	EVcRange* codeRange;
} EncVec;

/* Select encoding vector with respect to pid and eid */
EncVec* eVecMap[5][10];

static EVLow gbk_low_range[] = {{0x40, 0x7E}, {0x80, 0xFE}};
static EVcRange gbk_coderange = {{0x81, 0xFE}, 2, &gbk_low_range[0]};
static EncVec gbk_enc_vec = {1, &gbk_coderange};

/*************/
/* Functions */
/*************/



void mesg(const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	vfprintf(stderr, msg, ap);
	va_end(ap);
}



void fatal(const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	vfprintf(stderr, msg, ap);
	fprintf(stderr, "\n");
	va_end(ap);
	exit(FAILURE);
}


void fatal_error(const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	vfprintf(stderr, msg, ap);
	fprintf(stderr, "\n");
	va_end(ap);
	exit(FAILURE);
}

/*
 * Reallocate a pointer.
 */

void * re_alloc(void* ptr, size_t size, char* sub)
{
	register void* value = realloc(ptr, size);

	if (value == NULL)
		fatal("%s: Virtual memory exhausted", sub);
	return value;
}

/*
 * Open TTF file and select cmap.
 */

int Init_Font_Engine(char* infile)
{
	UShort cmapindex, platformID, encodingID, num_cmap;

	/* initialization of the FreeType engine */
	error = TT_Init_FreeType(&engine);
	if (error)
		fatal_error("Couldn't initialize FreeType engine");

	/* initialization of the post extension */
	error = TT_Init_Post_Extension(engine);
	if (error)
		fatal_error("Couldn't initialize the post extension");

	/* open the input file */
	error = TT_Open_Face(engine, infile, &face);
	if (error)
		fatal_error("Unable to open input file `%s'", infile);

	/* load full post table */
	error = TT_Load_PS_Names(face, &post);
	if (error)
		fatal_error("Unable to load post table");

	/* get face properties */
	TT_Get_Face_Properties(face, &properties);

	/* Load the instance. */
	error = TT_New_Instance(face, &instance);
	if (error)
		fatal_error("Couldn't create instance");

	error = TT_Set_Instance_Resolutions(instance, 600, 600);
	if (error)
		fatal_error("Error setting resolutions");

	error = TT_Set_Instance_CharSize(instance, 120 * 64);
	if (error)
		fatal_error("Error setting character size");

	error = TT_New_Glyph(face, &glyph);
	if (error)
		fatal_error("Couldn't create new glyph");

	/* Get the requested cmap. */
	num_cmap = TT_Get_CharMap_Count(face);
	for (cmapindex = 0; cmapindex < num_cmap; cmapindex++)
	{
		TT_Get_CharMap_ID(face, cmapindex, &platformID, &encodingID);
		if (platformID == TT_PLATFORM_MICROSOFT) {
			pid = platformID; eid = encodingID;
			break;
		}
	}
	/* malloc for glyph data */
	error = TT_Get_CharMap(face, cmapindex, &cmap);
	if (error)
		fatal_error("Cannot load cmap");
	return TRUE;
}


/*
 * Get GBK/BIG5 encode form Unicode TTF files.
 */

int getIndex(UShort char_Code)
{
	int idx, pos;
	UShort unicode;
	pos = ((char_Code>>8)-0x81)*192 + ((char_Code&0x00FF)-0x40);
	unicode = gbk2uni[pos];
	idx = TT_Char_Index (cmap, unicode);
	return idx;
}



/*
 * Get font infos: name, version, copyright.
 */

char* LookUp_Name(int index)
{
	UShort platform, encoding, language, id;
	char*  string;
	UShort string_len;
	UShort i, n;

	n = properties.num_Names;

	for (i = 0; i < n; i++)
	{
		TT_Get_Name_ID(face, i, &platform, &encoding, &language, &id);
		TT_Get_Name_String(face, i, &string, &string_len);

		if (id == index)
			break;
	}
	i = (string_len > NAMELEN) ? NAMELEN : string_len;
	string[i] = '\0';
	return string;
}


/*
 * Load a glyph's outline and metrics.
 */

int LoadTrueTypeChar(int idx)
{
	TT_Matrix scale = {(1 << 16) / 64, 0, 0, (1 << 16) / 64};

	error = TT_Load_Glyph(instance, glyph, idx, TTLOAD_DEFAULT);
	if (error)
		fatal_error("Load glyph");

	error = TT_Get_Glyph_Outline(glyph, &outline);
	if (error)
		fatal_error("Get glyph outlines");

	TT_Transform_Outline(&outline, &scale);

	error = TT_Get_Glyph_Metrics(glyph, &metrics);
	if (error)
		fatal_error("Get glyph_metrics");

	return TRUE;
}


/*
 * Get PS name of a glyph.
 */

char* PS_GlyphName(UShort idx, UShort code)
{
	char *glyphname = ".notdef";
	static char CJK_glyphname[8];
	sprintf(CJK_glyphname, "cjk%04X", code);
	glyphname = CJK_glyphname;
	return glyphname;
}


/*
 * Header of Type 1 font.
 */

char      fontname[NAMELEN]; 
char      fam_name[NAMELEN];
void PS_Head(FILE *out, int plane, EncVec* planeEV, int UID)
{
	EVcRange* cRange = planeEV->codeRange;
	UShort    numCR  = planeEV->numCodeRanges;
	int       cjk = 0, nGlyph = 0, irange;
	EVLow*    pLow = cRange->low;
	UShort    nLow = cRange->numLowRanges;
	int       ipl, ilow, ich;
	int       idx;
	UShort    code;
	time_t    curtime;
	struct tm *loctime;
	char      text[NAMELEN];
	char*     version;
	char      copyright[NAMELEN];

	/* Get the current time with local representation */
	curtime = time(NULL);
	loctime = localtime(&curtime);


	strcpy(text, LookUp_Name(5));
	version = &text[strcspn(text, "1234567890.")];
	version[strspn(version, "1234567890.")] = '\0';
	strcpy(copyright, LookUp_Name(0));

	fprintf(out, "%%!FontType1-1.0: %s %s\n", fontname, version);
	fprintf(out, "%%%%Creator: %s, ", prog);
	fprintf(out, "%s\n", rcsid);
	fprintf(out, "%%%%CreationDate: %s", asctime(loctime));

	Fputs("%%VMusage: 030000 030000");
	Fputs("11 dict begin");
	Fputs("/FontInfo 8 dict dup begin");
	fprintf(out, "/version (%s) readonly def\n", version);
	fprintf(out, "/Copyright (%s) readonly def\n", copyright);
	fprintf(out, "/Notice (Plane %d) readonly def\n", plane);
	fprintf(out, "/FullName (%s%02d) readonly def\n",fontname,plane);
	fprintf(out, "/FamilyName (%s) readonly def\n",fam_name);
	Fputs("/Weight (Regular) readonly def");
	Fputs("/ItalicAngle 0 def");
	Fputs("/isFixedPitch false def");
	/* Fputs("/UnderlineThickness 50 def"); */
	Fputs("end readonly def");
	fprintf(out, "/FontName /%s%02d def\n",fontname,plane);
	Fputs("/PaintType 0 def");
	Fputs("/FontType 1 def");
	Fputs("/FontMatrix [0.001 0 0 0.001 0 0] readonly def");
	Fputs("/Encoding 256 array");
	Fputs("0 1 255 {1 index exch /.notdef put} for");

	/* encoding vector */
	for (irange = 0; irange < numCR; irange++, cRange++)
	{
		pLow = cRange->low;
		nLow = cRange->numLowRanges;
		for (ipl = cRange->high.start; ipl <= cRange->high.end; ipl++)
		{
			if (nLow == 0)
			{
				nGlyph = 0x100;
				for (ich = 0; ich <= 0xff; ich++)
				{
					code = ipl<<8 | ich;
					/*          idx = TT_Char_Index(cmap, code); */
					code_idx[ich] = idx = getIndex(code);
					fprintf(out, "dup %d /%s put\n", ich, PS_GlyphName(idx, code));
				}
			}
			else
			{
				for (ilow = 0; ilow < nLow; ilow++, pLow++)
				{
					nGlyph += pLow->end - pLow->start + 1;
					for (ich = pLow->start; ich <= pLow->end; ich++, cjk++)
					{
						code = ipl<<8 | ich;
						/* idx = TT_Char_Index(cmap, code); */
						code_idx[cjk] = idx = getIndex(code);
						fprintf(out, "dup %d /%s put\n", cjk, PS_GlyphName(idx, code));

					}
				}
			}
		}
	}

	Fputs("readonly def");
	Fputs("/FontBBox [0 -300 1000 1000] readonly def");
	fprintf(out, "/UniqueID %d def\n",UID);
	Fputs("currentdict end");
	Fputs("currentfile eexec");

	Fputs("dup /Private 8 dict dup begin");
	Fputs("/-| { string currentfile exch readstring pop } executeonly def");
	Fputs("/|- { noaccess def } executeonly def");
	Fputs("/| { noaccess put } executeonly def");
	Fputs("/BlueValues [ ] |-");
	Fputs("/ForceBold true def");
	Fputs("/LanguageGroup 1 def");
	Fputs("/RndStemUp false def");
	Fputs("/MinFeature{16 16} |-");
	/* Fputs("/password 5839 def"); */
	fprintf(out, "/UniqueID %d def\n",UID);

	Fputs("/Subrs 4 array");
	Fputs("dup 0 { 3 0 callothersubr pop pop setcurrentpoint return } |");
	Fputs("dup 1 { 0 1 callothersubr return } |");
	Fputs("dup 2 { 0 2 callothersubr return } |");
	Fputs("dup 3 { return } |");
	Fputs("|-");

	fprintf(out, "2 index /CharStrings %d dict dup begin\n", nGlyph + 1);
}


/*
 * Tail of Type 1 font.
 */

void PS_Tail(FILE *out)
{
	Fputs("/.notdef { 0 250 hsbw endchar } |-");
	Fputs("end end readonly put noaccess put");
	Fputs("dup /FontName get exch definefont pop");
	Fputs("mark currentfile closefile");
}


/*
 * Use the `rrcurveto' command on more than one `off' points.
 */

void PS_Curveto(FILE *out, TT_F26Dot6 x, TT_F26Dot6 y, int s, int e)
{
	int  N, i;
	TT_F26Dot6 sx[3], sy[3], cx[4], cy[4];

	N = e - s + 1;
	cx[0] = lastpsx; cy[0] = lastpsy;
	if (s == e)
	{
		cx[1] = (2 * outline.points[s].x + outline.points[s - 1].x) / 3;
		cy[1] = (2 * outline.points[s].y + outline.points[s - 1].y) / 3;
		cx[2] = (2 * outline.points[s].x + x) / 3;
		cy[2] = (2 * outline.points[s].y + y) / 3;
		cx[3] = x;
		cy[3] = y;

		fprintf(out, "%d %d %d %d %d %d rrcurveto\n",
				Coord(cx[1] - cx[0]), Coord(cy[1] - cy[0]),
				Coord(cx[2] - cx[1]), Coord(cy[2] - cy[1]),
				Coord(cx[3] - cx[2]), Coord(cy[3] - cy[2]));
	}
	else
	{
		for(i = 0; i < N; i++)
		{
			sx[0] = (i == 0) ?
				outline.points[s - 1].x :
				(outline.points[i + s].x + outline.points[i + s - 1].x) / 2;
			sy[0] = (i == 0) ?
				outline.points[s - 1].y :
				(outline.points[i + s].y + outline.points[i + s - 1].y) / 2;
			sx[1] = outline.points[s + i].x;
			sy[1] = outline.points[s + i].y;
			sx[2] = (i == N - 1) ?
				x :
				(outline.points[s + i].x + outline.points[s + i + 1].x) / 2;
			sy[2] = (i == N - 1) ?
				y :
				(outline.points[s + i].y + outline.points[s + i + 1].y) / 2;

			cx[1] = (2 * sx[1] + sx[0]) / 3;
			cy[1] = (2 * sy[1] + sy[0]) / 3;
			cx[2] = (2 * sx[1] + sx[2]) / 3;
			cy[2] = (2 * sy[1] + sy[2]) / 3;
			cx[3] = sx[2];
			cy[3] = sy[2];

			fprintf(out, "%d %d %d %d %d %d rrcurveto\n",
					Coord(cx[1] - cx[0]), Coord(cy[1] - cy[0]),
					Coord(cx[2] - cx[1]), Coord(cy[2] - cy[1]),
					Coord(cx[3] - cx[2]), Coord(cy[3] - cy[2]));

			cx[0] = cx[3];
			cy[0] = cy[3];
		}
	}
	PS_LastPt(x, y);
}


void writeEncFile(char *enc_name)
{
	int i;
	FILE  *idx_file;
	if ((idx_file = fopen(enc_name, "wt")) == 0)
		fatal("%s: unable to open the encoding file '%s' for writing.\n",
				prog, enc_name);
	fprintf(idx_file, "/%sEncoding [\n", enc_name);
	for (i = 0; i < 256; i++)
		if (code_idx[i] == 0)
			fprintf(idx_file, "/.notdef\n");
		else
			fprintf(idx_file, "/index0x%X\n", code_idx[i]);
	fprintf(idx_file, "] def\n");
	fclose(idx_file);
}


/*
 * Construct CharString of a glyph.
 */

short PS_CharString(FILE *out, UShort char_Code)
{
	int    idx, i, j;
	UShort start_offpt, end_offpt = 0, fst;


	/*  idx = TT_Char_Index(cmap, char_Code); */
	idx = getIndex(char_Code);
	if (idx == 0) {
		fprintf(out, "/cjk%04X { 0 250 hsbw endchar } |-\n", char_Code);
		return FALSE;
	}

	if (!LoadTrueTypeChar(idx))
		fatal("Couldn't load character with index %d (code %d)", idx, char_Code);

	/* Begin string */
	fprintf(out, "/%s {\n", PS_GlyphName(idx, char_Code));


#ifdef DEBUG
	if (char_Code == debug_Char_Code)
	{
		tmp_out(tmpout);
		out  = tmpout;
		loop = 0;
	}
	for (; loop < 2; loop++)
	{
#endif

		/* coordinates are all relative to (0,0) in FreeType */
		fprintf(out, "0 %d hsbw\n", (int)(metrics.advance / 64));

		/* Initialize ending contour point, relative coordinates */
		lastpsx = lastpsy = 0;

		for (i = 0, j = 0; i < outline.n_contours; i++)
		{
			fst = j;
			PS_Moveto(outline.points[j].x, outline.points[j].y);
			j++;

			start_offpt = 0; /* start at least 1 */

			/*
			 * data pts for all contours stored in one array.
			 * each round j init at last j + 1
			 */

			/*
			 * start_offpt means start of off points.
			 * 0 means no off points in record.
			 * N means the position of the off point.
			 * end_offpt means the ending off point.
			 * lastx, lasty is the last ON point from which Curve and Line
			 * shall start.
			 */

			/*
			 * start with j=0. into loop, j=1.
			 * if pt[1] off, if start_offpt == 0, toggle start_offpt
			 * next j=2. if on, now start_off != 0, run Curveto.
			 * if pt[1] on, start_off == 0, will run Lineto.
			 */

			for (; j <= outline.contours[i]; j++)
			{
				if (!(outline.flags[j] & TT_Flag_On_Curve))
				{
					if (!start_offpt)
						start_offpt = end_offpt = j;
					else
						end_offpt++;
				}
				else
				{                    /* On Curve */
					if (start_offpt)
					{
						/*
						 * start_offpt stuck at j, end_offpt++.
						 * end_offpt - start_offpt gives no of off pts.
						 * start_offpt gives start of sequence.
						 * why need outline.xCoord[j] outline.yCoord[j]?
						 */

						PS_Curveto(out,
								outline.points[j].x, outline.points[j].y,
								start_offpt, end_offpt);
						start_offpt = 0;

						/*
						 * also use start_offpt as indicator to save one variable!!
						 * after curveto, reset condition.
						 */
					}
					else
						PS_Lineto(outline.points[j].x, outline.points[j].y);
				}
			}

			/*
			 * looks like closepath fst = first, i.e. go back to first
			 */

			if (start_offpt)
				PS_Curveto(out,
						outline.points[fst].x, outline.points[fst].y,
						start_offpt, end_offpt);
			else
				Fputs("closepath");
		}

		Fputs("endchar");

#if DEBUG
		out = oldout;
	}
	if (char_Code == debug_Char_Code)
	{
		if (showlabel && !no_glyph)
			Fputps("drawlabel");
		else
			Fputps("nodrawlabel");
		Fputps("");
		Fputps("%%% end of drawing");
		Fputps("oldgray setgray");
		Fputps("showpage");
		fclose(tmpout);
	}
#endif

	Fputs(" } |-");
	return TRUE;
}

/*
 * Match code ranges by a font plane.
 */

EncVec*
Get_PlaneEV(EncVec* encVec,
		int plane)
{
	UShort    numCR  = encVec->numCodeRanges;
	EVcRange* cRange = encVec->codeRange;

	EncVec*   encV = NULL;
	EVcRange* planeCR = NULL;
	EVLow*    planeLow = NULL;
	UShort    nCR = 0, nLow = 0;

	int       icr;

	int iChar = 0;               /* summed # of chars */
	int nChar = (plane-1) * 256; /* the first char code ranges recorded */
	int recording = 0;


	/* if compact, plane starts from 1 to be */
	/* compatible with the CJK package */
	if (plane < 1 || plane > 99)
		fatal("Get_PlaneEV: Given plane out of range");

	for (icr = 0; icr < numCR; icr++, cRange++)
	{
		UShort numLow = cRange->numLowRanges;
		int    ipl;

		for (ipl = cRange->high.start; ipl <= cRange->high.end; ipl++)
		{
			EVLow* pLow = cRange->low;
			int ilow;

			if (recording)
			{ /* if we have made a hit */
				if (planeLow != NULL)
				{ /* if low byte range has not been saved */
					(planeCR + nCR - 1)->low = planeLow;
					(planeCR + nCR - 1)->numLowRanges = nLow;
					planeLow = NULL;
				}

				/* each new plane starts a EVcRange if */
				/* iChar is still less than nChar */
				if (iChar <= nChar)
				{
					planeCR = re_alloc(planeCR, ++nCR * sizeof (EVcRange),
							"Get_PlaneEV");
					(planeCR + nCR - 1)->high.start =
						(planeCR + nCR - 1)->high.end = ipl;
					(planeCR + nCR - 1)->numLowRanges = nLow = 0;
				}
			}

			/* scan each low byte range */
			for (ilow = 0; ilow < (numLow == 0 ? 1 : numLow); ilow++, pLow++)
			{
				int start, end, nLowChar;

				if (numLow == 0)
				{ /* default range */
					start = 0x0;
					end   = 0xff;
				}
				else
				{
					start = pLow->start;
					end   = pLow->end;
				}
				nLowChar = end - start + 1;
				if (iChar + nLowChar > nChar)
				{ /* a hit! */
					int bchar = start + nChar - iChar;
					if (planeCR == NULL)
					{
						/* the first time code range is recorded */
						planeCR = re_alloc(planeCR, ++nCR * sizeof (EVcRange),
								"Get_PlaneEV");
						(planeCR + nCR - 1)->high.start = ipl;
						(planeCR + nCR - 1)->high.end   = ipl;
					}

					/* adjust range boundary */
					if (recording == 0)
						start = bchar;
					else
						end = bchar;
					nChar += 0xff;

					/* recording starts */
					recording++;
				}

				iChar += nLowChar;    /* next range */

				if (recording)
				{
					/* a new low range */
					if (iChar <= nChar)
					{
						planeLow = re_alloc(planeLow, ++nLow * sizeof (EVLow),
								"Get_PlaneEV");
						(planeLow + nLow - 1)->start = start;
						(planeLow + nLow - 1)->end   = end;
					}
					if (recording > 1 || iChar > nChar)
					{
						/* beyond recording range */
						(planeCR + nCR - 1)->numLowRanges = nLow;
						(planeCR + nCR - 1)->low = planeLow;
						encV = re_alloc(encV, 1 * sizeof (EncVec), "Get_PlaneEV");
						encV->numCodeRanges = nCR;
						encV->codeRange = planeCR;
						return encV;
					}
				}
			}
		}
	}
	/* we must finalize the ranges */
	if (recording)
	{
		(planeCR + nCR - 1)->numLowRanges = nLow;
		(planeCR + nCR - 1)->low = planeLow;
		encV = re_alloc(encV, 1 * sizeof (EncVec), "Get_PlaneEV");
		encV->numCodeRanges = nCR;
		encV->codeRange = planeCR;
		return encV;
	}
	return NULL;
}


/*
 * The main subroutine for generating Type 1 fonts.
 * One subfont per call.
 */

short
Generate_Font(FILE *out, int plane, int UID)

{
	EncVec*   encVec = &gbk_enc_vec;

	EncVec*   planeEncVec;
	EVcRange* cRange;
	UShort    numCR;
	UShort    code;
	int       ilow, iplan, ichar, irange;

	if (encVec == NULL)
		return FALSE;
	if ((planeEncVec = Get_PlaneEV(encVec, plane)) == NULL)
	{
		mesg("%s: Can't find encoding vector for the font plane 0x%X.\n",
				prog, plane);
		return FALSE;
	}

	/* Header of Type1 font */
	PS_Head(out, plane, planeEncVec, UID);

	numCR  = planeEncVec->numCodeRanges;
	cRange = planeEncVec->codeRange;

	for (irange = 0; irange < numCR; irange++, cRange++)
	{
		EVLow* pLow = cRange->low;
		UShort nLow = cRange->numLowRanges;

		for (iplan = cRange->high.start; iplan <= cRange->high.end; iplan++)
		{
			if (nLow == 0)
			{
				for (ichar = 0; ichar <= 0xff; ichar++)
				{
					code = iplan << 8 | ichar;
					PS_CharString(out, code);
				}
			}
			else
			{
				for (ilow = 0; ilow < nLow; ilow++, pLow++)
				{
					for (ichar = pLow->start; ichar <= pLow->end; ichar++)
					{
						code = iplan << 8 | ichar;
						PS_CharString(out, code);
					}
				}
			}
		}
	}
	PS_Tail(out);
	return TRUE;
}


/*
 * Main: process options, file I/O, etc.
 */

int
ttf2pfb(char *infile, 
	char *outfile, 
	char *enc_file, 
	int plane, 
	int UID)
{
	FILE *out;
	int  result, i;

	if (UID == 0) UID = 4999999;

        for (i = 0; i < 256; i++)
          code_idx[i] = 0;

	if ((out = fopen(outfile, "w")) == 0)
	{
		fatal("Unable to open the output file `%s'", outfile);
		exit(FAILURE);
	}

	if (!Init_Font_Engine(infile))
		fatal("can't open TrueType engine\n");

	/* Generate the disassembled PFB font from the TrueType font */
	if (! Generate_Font(out, plane, UID)) {
		mesg("An error occurred while generating pfb font\n");
		unlink(outfile);
		exit (-1);
	} 

	fclose(out);

	TT_Close_Face(face);
	TT_Done_FreeType(engine);

	writeEncFile(enc_file);
	return 0;
}

